// Boids.js

function buildUI( obj ) {
  
  obj.setParameter("name", "Boids");
  
  obj.addParameterFloat("timer", 0, 0, 10000, true, true);
  obj.addParameterLink("base particle", true);
  
  obj.addParameterSeparator("Boid");
  
  obj.addParameterBool("separation", 1, 0, 1, true, true);
  obj.addParameterFloat("separation limit", 2, 0, 10000, true, true);
  obj.addParameterBool("alignment", 1, 0, 1, true, true);
  obj.addParameterFloat("alignment limit", 10, 0, 10000, true, true);
  obj.addParameterFloat("coherent limit", 20, 0, 10000, true, true);
  obj.addParameterBool("cohere", 1, 0, 1, true, true);
  obj.addParameterFloat("speed", 1, 0, 10000, true, true); 
  obj.addParameterFloat("max speed", 3, 0, 10000, true, true);
  
  obj.addParameterLink("attractions", true);
  obj.addParameterLink("repulsions", true);
  
  obj.addParameterLink("obstacles", true);
  obj.addParameterFloat("avoid angle", 45, 10, 90, true, true);
  
  obj.addParameterSeparator("Gage");
  
  obj.addParameterLink("gage", true);  
}

var lastAnimPosition = 0;

var particles = [];
var distances = [];

function buildObject( obj ) {
  
  var doc = obj.document();// get document
  var core = obj.core(); // get particlecore
  
  var base = obj.getParameter("base particle");
  
  var separationLimit = obj.getParameter("separation limit");
  var alignmentLimit = obj.getParameter("alignment limit");
  var coherentLimit = obj.getParameter("coherent limit");
  
  var speedFac = obj.getParameter("speed");
  var maxSpeed = obj.getParameter("max speed");
  
  var tglCohere = obj.getParameter("cohere");
  var tglSeparation = obj.getParameter("separation");
  var tglAlignment = obj.getParameter("alignment");
  
  var gageObj = obj.getParameter("gage");
  var gage = getBoundingBox( gageObj );
  
  var gageMax = gage.max;
  var gageMin = gage.min;
  
  //
  var attractions = obj.getParameter("attractions");
  var repulsions = obj.getParameter("repulsions");
  
  var obstacles = obj.getParameter("obstacles");
  var avoidAngle = obj.getParameter("avoid angle");
  
  if (!base || base.family() != PARTICLEFAMILY) return;
  
  var baseCore = base.core();
  
  var count = particles.length;
  
  var baseCount = baseCore.particleCount();
  var animPosition = doc.animPosition();
  
  if (animPosition == 0 || count != baseCount) { // reset all particles
    
    //print( 'init:' + animPosition + ', baseCount:' + baseCount );
    
    particles.length = 0;
    
    for (var i = 0;i < baseCount;i++) {
    
      var baseParticle = baseCore.particleAtIndex( i );
      
      particles.push( new Boid( baseParticle.getPosition(),
                                new Vec3D( 0.5 - Math.random(), 0.5 - Math.random(), 0.5 - Math.random() ) ) );
    }
    
    lastAnimPosition = animPosition;
  }
  
  calcDistances();
  
  var avoidCounter = Math.ceil( 360 / avoidAngle );
  var avoidMat_h = new Mat4D( ROTATE_HPB, avoidAngle, 0, 0 );
  var avoidMat_p = new Mat4D( ROTATE_HPB, 0, avoidAngle, 0 );
  var avoidMat_b = new Mat4D( ROTATE_HPB, 0, 0, avoidAngle );
  
  var avoidObstacle = function( _sp, _mat ) {

    //return _sp.inverse();
    return _mat.multiply( _sp );
  };
  
  var childCount = 0;
  var attr_list = [];
  if (attractions && attractions.type() == FOLDER) {
    childCount = attractions.childCount();
    for (var ai = 0;ai < childCount;ai++) {
      attr_list.push( attractions.childAtIndex( ai ) );
    }
  } else {
    if (attractions) attr_list.push( attractions );
  }
  
  var repul_list = [];
  if (repulsions && repulsions.type() == FOLDER) {
    childCount = repulsions.childCount();
    for (var ri = 0;ri < childCount;ri++) {
      repul_list.push( repulsions.childAtIndex( ri ) );
    }
  } else {
    if (repulsions) repul_list.push( repulsions );
  }
    
  var ob_list = [];
  if (obstacles && obstacles.type() == FOLDER) {
    childCount = obstacles.childCount();
    for (var oi = 0;oi < childCount;oi++) {
      ob_list.push( obstacles.childAtIndex( oi ) );
    }
  } else {
    if (obstacles) ob_list.push( obstacles );
  }
  
  var ob_len = ob_list.length;
  
  //
  
  for (var i = 0;i < baseCount;i++) {
    
    var boid = particles[i];
    var particle = core.addParticle();
    
    var pos = boid.pos;
    var pos_tmp = pos.copy();
    var speed = boid.speed;
    
    var cohere = new Vec3D();
    var separation = new Vec3D();
    
    var alignment = new Vec3D();
    
    var cohereCount = 0;
    var separationCount = 0;
    
    var alignmentCount = 0;
    
    for (var j = 0;j < baseCount;j++) {
      if ( i != j ) {
        var target = particles[j];
        var d;
        var distance = distances[i][j];
        
        if (distance < separationLimit) {
          separation = separation.add( target.pos );
          separationCount++;
        } else if (distance < alignmentLimit) {
          alignment = alignment.add( target.pos );
          alignmentCount++;
        } else if (distance < coherentLimit) {
          cohere = cohere.add( target.pos );
          cohereCount++;
        }
      }
    }
    
    if (tglCohere && cohereCount) {
      cohere = Vec3D_normalize( ( cohere.multiply( 1 / cohereCount ) ).sub( pos ) );
      speed = speed.sub( cohere );
    }
    if (tglAlignment && alignmentCount) {
      alignment = Vec3D_normalize( ( alignment.multiply( 1 / alignmentCount ) ).sub( pos ) );
      speed = speed.add( alignment );
    }
    if (tglSeparation && separationCount) {
      separation = Vec3D_normalize( ( separation.multiply( 1 / separationCount ) ).sub( pos ) );
      speed = speed.add( separation );
    }
    
    if (repul_list.length > 0) {
      var len = repul_list.length;
      for (var ri = 0;ri < len;ri++) {
        var rep = repul_list[ri];
        var tar = rep.objMatrix().multiply( new Vec3D() );
        var power = (rep.family() == FORCEFAMILY && rep.parameterWithName('strength'))? rep.getParameter("strength") : 5;
        
        var dist = Vec3D_distance( tar, pos );
        if (dist < power) {
          speed = speed.add( Vec3D_normalize( tar.sub( pos ) )  );
        }
      }
    }
    
    if (attr_list.length > 0) {
      var len = attr_list.length;
      for (var ai = 0;ai < len;ai++) {
        var attr = attr_list[ai];
        var tar = attr.objMatrix().multiply( new Vec3D() );
        var power = (attr.family() == FORCEFAMILY && attr.parameterWithName('strength'))? attr.getParameter("strength") : 5;
        
        var dist = Vec3D_distance( tar, pos );
        if (dist < power) {
          speed = speed.sub( Vec3D_normalize( tar.sub( pos ) ) );
        }
      }
    }
    
    if ( speed.norm() > maxSpeed ) speed = speed.multiply( 1 / speed.norm() * maxSpeed );
    
    if (obstacles) {
      pos_tmp = pos.add( speed.multiply( (lastAnimPosition - animPosition) * speedFac ) );
      ob_check = true;
      ob_count = 0;
      
      do {
        ob_check = true;
        
        if (ob_count < avoidCounter) {
          avoidMat = avoidMat_h;
        } else if (ob_count < avoidCounter * 2) {
          avoidMat = avoidMat_p;
        } else {
          avoidMat = avoidMat_b;
        }
        
        for (var oi = 0;oi < ob_len;oi++) {
          var ob = ob_list[oi];
          var ob_type = ob.type();
          var ob_mat = ob.obj2WorldMatrix();
          
          var ob_pos = ob_mat.inverse().multiply( pos_tmp );
          var ob_dif = new Vec3D();
          
          switch( ob_type ) {
            case BOX:
              var sizeX = ob.getParameter("sizeX") / 2;
              var sizeY = ob.getParameter("sizeY") / 2;
              var sizeZ = ob.getParameter("sizeZ") / 2;
              
              if ( (ob_pos.x > - sizeX && ob_pos.x < sizeX) &&
                   (ob_pos.y > - sizeY && ob_pos.y < sizeY) &&
                   (ob_pos.z > - sizeZ && ob_pos.z < sizeZ) ) {
                ob_check = false;
              }
      
              break;
            case BALL:
              var radius = ob.getParameter("radius");
              
              if ( Vec3D_distance( ob_pos ) < radius ) {
                ob_check = false;
              }
              break;
            case CYLINDER:
              var radius = ob.getParameter("radius");
              var height = ob.getParameter("height") / 2;
              
              if ( (ob_pos.y > - height && ob_pos.y < height) &&
                   ( Vec3D_distance( new Vec3D( ob_pos.x, 0, ob_pos.z ) ) < radius ) ) {
                ob_check = false;
              }
              break;
           }
         }
         
         if (ob_check == false) {
           speed = avoidObstacle( speed, avoidMat );     
           pos_tmp = pos.add( speed.multiply( ( lastAnimPosition - animPosition ) * speedFac ) );
           
           if (ob_count++ > avoidCounter * 3) {
             //print( 'not found' );
             ob_check = true;
           }
         }
       } while ( ob_check == false );
    }
    
    pos = pos.add( speed.multiply( (lastAnimPosition - animPosition) * speedFac ) );
    

    if (gageObj) {
      if (pos.x < gageMin.x) { pos.x = gageMin.x; speed.x *= -1 }
      else if (pos.x > gageMax.x) { pos.x = gageMax.x; speed.x *= -1 }
      
      if (pos.y < gageMin.y) { pos.y = gageMin.y; speed.y *= -1 }
      else if (pos.y > gageMax.y) { pos.y = gageMax.y; speed.y *= -1 }
      
      if (pos.z < gageMin.z) { pos.z = gageMin.z; speed.z *= -1 }
      else if (pos.z > gageMax.z) { pos.z = gageMax.z; speed.z *= -1 }
    }
    
    
    // update particles
    particles[i].pos = pos;
    particles[i].speed = speed;
    
    particles[i].calcRotation();
    
    particle.setPosition( pos );
    particle.setRotation( boid.rotation );
    
    particle.setUVW( speed );
  }
  
  lastAnimPosition = animPosition;
}

function calcDistances() {
  var i, j;
  var len = particles.length;
  for (i = 0;i < len;i++) {
    distances[i] = [];
  }
  
  for (i = 0;i < len;i++) {
    var ba = particles[i];
    
    for (j = i;j < len;j++) {
      var d = 0;
      if (i != j) {
        var bb = particles[j];
        d = Vec3D_distance( ba.pos, bb.pos );
      }
      distances[ i ][ j ] = d;
      distances[ j ][ i ] = d;
      
    }
  }
}

function getBoundingBox( obj ) {
  
  if (!obj || obj.family() != NGONFAMILY) {
    return { max: new Vec3D(  20,  20,  20 ), min: new Vec3D( -20, -20, -20 ) }
  }
  
  var mat = obj.objMatrix();
  var core = obj.modCore();
  var vertexCount = core.vertexCount();
  var max, min;
  for (var i = 0;i < vertexCount;i++) {
    var vec = mat.multiply( core.vertex(i) );
    if (i == 0) {
      max = vec.copy();
      min = vec.copy();
    } else {
      max.x = Math.max( max.x, vec.x );
      max.y = Math.max( max.y, vec.y );
      max.z = Math.max( max.z, vec.z );
      
      min.x = Math.min( min.x, vec.x );
      min.y = Math.min( min.y, vec.y );
      min.z = Math.min( min.z, vec.z );
    }
  }
  
  return { max: max, min: min };
}

// boid 

var RAD2DEG = 180 / Math.PI;

function Boid( pos, speed ) {
  this.pos = pos;
  this.speed = speed;
  this.rotation = new Vec3D();
  
}

Boid.prototype.calcRotation = function() {
  var vec = Vec3D_normalize( this.speed );
  
  var theta = Math.acos( vec.y ) * RAD2DEG;
  var phi = Math.atan2( vec.x, vec.z ) * RAD2DEG;
  
  this.rotation.x = theta;
  this.rotation.y = phi;
}

// helper

function objectWithType(obj, type) {
    
}

var Vec3D_distance = function() {
	if( arguments.length == 1)
			return Math.sqrt( arguments[0].x*arguments[0].x + arguments[0].y*arguments[0].y + arguments[0].z*arguments[0].z );
	var p = arguments[1].sub(arguments[0]);
	return Math.sqrt( p.x*p.x + p.y*p.y + p.z*p.z );
}

var Vec3D_normalize = function(vec) {
  var l = vec.norm();
  if (l != 0) {
    return vec.multiply( 1/l );
  }
  return vec;
}
