//
// Stepper.js
//
// required version : Cheetah3D v.4.0 above
//
//  (c) Hiroto Tsubaki
//  http://www.tres-graficos.jp/
//  tg@tres-graficos.jp
//
//  2006-12-11 created.
//  2006-12-12 bug fixed. add capture button.
//  2006-12-13 fixed some code.add parameter "selection", fix bug.
//  2007-01-25 use new func.
//  2012-05-04 undo/redo support.
//  2012-05-16 added 'set value' function
//
// Usage: Place this into ~/Library/Application Support/Cheetah3D/scripts/Tool folder. restart Cheetah3D, then select from Tools -> Scritp -> Tool Script
//        this Tool script for move/rotate/scale selected points with input values.
//

function buildUI(tool) {
    tool.addParameterSeparator("Stepper");
    
    if (!tool.document().editMode) tool.addParameterSelector("selection",["point","edge","polygon"],true,true);
    
    tool.addParameterBool("set x",1,0,1,true,true);
    tool.addParameterFloat("step x",0,-1000,1000,true,true);
    tool.addParameterBool("set y",1,0,1,true,true);
    tool.addParameterFloat("step y",0,-1000,1000,true,true);
    tool.addParameterBool("set z",1,0,1,true,true);
    tool.addParameterFloat("step z",0,-1000,1000,true,true);
    
    tool.addParameterSeparator("Operations");
    tool.addParameterSelector("coordinate system",["selection","selection normal","object","world","input point"],true,true);
    tool.addParameterButton("step move","move","Move");
    tool.addParameterButton("step rotate","rotate","Rotate");
    tool.addParameterButton("step scale","scale","Scale");
    
    tool.addParameterSeparator("Input Point (rotate/scale center)");
    tool.addParameterFloat("point x",0,-1000,1000,true,true);
    tool.addParameterFloat("point y",0,-1000,1000,true,true);
    tool.addParameterFloat("point z",0,-1000,1000,true,true);
    tool.addParameterButton("capture input point","capture","Capture");
    
    tool.addParameterSeparator("Set Value");
    tool.addParameterFloat("value x", 0, -10000, 10000, true, true);
    tool.addParameterFloat("value y", 0, -10000, 10000, true, true);
    tool.addParameterFloat("value z", 0, -10000, 10000, true, true);
    tool.addParameterButton("set value", "set", "setValue");
    tool.addParameterButton("capture value", "capture", "captureValue");
    
    captureValue( tool );
}

function Set50(tool) {
    tool.setParameter("step x",0.5);
    tool.setParameter("step y",0.5);
    tool.setParameter("step z",0.5);
}

function Set100(tool) {
    tool.setParameter("step x",1.0);
    tool.setParameter("step y",1.0);
    tool.setParameter("step z",1.0);
}

function Set200(tool) {
    tool.setParameter("step x",2.0);
    tool.setParameter("step y",2.0);
    tool.setParameter("step z",2.0);
}

function Invert(tool) {
    var stepx = tool.getParameter("step x");
    var stepy = tool.getParameter("step y");
    var stepz = tool.getParameter("step z");
    
    tool.setParameter("step x",stepx*-1);
    tool.setParameter("step y",stepy*-1);
    tool.setParameter("step z",stepz*-1);
}

function Capture(tool) {
    var doc = tool.document();
    var obj = doc.selectedObject();
    
    if (obj.family() == SPLINEFAMILY || obj.family() == LIGHTFAMILY || obj.family() == MODIFIERFAMILY || obj.family() == NGONFAMILY) {
        var position = obj.getParameter("position");
        tool.setParameter("point x", position.x);
        tool.setParameter("point y", position.y);
        tool.setParameter("point z", position.z);
        
        tool.update();
    }
}

function getCenterVecForSelector(coor, list, obj, tool) {
    var i;
    var len = list.length;
    var core = obj.core();
    var centerVec = new Vec3D(0,0,0);
    switch (coor) {
        case 0: // selected
            for (i = 0;i < len;i++) {
                var vec = core.vertex(list[i]);
                centerVec = centerVec.add(vec);
            }
            centerVec = centerVec.multiply(1/len);
            break;
        case 1: // selected normal
            for (i = 0;i < len;i++) {
                var vec = core.vertex(list[i]);
                centerVec = centerVec.add(vec);
            }
            centerVec = centerVec.multiply(1/len);
            break;
        case 2: // object
            break;
        case 3: // world
            break;
        case 4: // captured point
            centerVec = new Vec3D(tool.getParameter("point x"), tool.getParameter("point y"), tool.getParameter("point z"));
            break;
        default:
            //print("error");
    }
    //print("coor:"+coor);
    //printVec3D(centerVec);
    return centerVec;
}

var normal;

function getVertexList(core, tool) {
    var doc = tool.document();
    var mode = (doc.editMode)? doc.editMode() : tool.getParameter("selection");
    var list = new Array();
    var i;
    
    var normals = new Array();
    
    switch(mode) {
        case POINT_MODE: // point
            var vCount = core.vertexCount();
            for (i = 0;i < vCount;i++) {
                if (core.vertexSelection(i)) {
                    list.push(i);
                }
            }
            var pCount = core.polygonCount();
            for (i = 0;i < pCount;i++) {
              var pSize = core.polygonSize(i);
              var j;
              for (j = 0;j < pSize;j++) {
                if (list.indexOf( core.vertexIndex(i, j) ) > -1) {
                  normals.push( core.normal(i, j ) );
                }
              }
            }
            break;
        case EDGE_MODE: // edge
            var pCount = core.polygonCount();
            for (i = 0;i < pCount;i++) {
                var pSize = core.polygonSize(i);
                var j;
                for (j = 0;j < pSize;j++) {
                    if (core.edgeSelection(i,j,SELECT)) {
                        var point1 = j;
                        var point2 = (j+1==pSize)? 0 : j+1;
                        var vIndex1 = core.vertexIndex(i,point1);
                        var vIndex2 = core.vertexIndex(i,point2);
                        if (! isExistIndex(list, vIndex1)) list.push(vIndex1);
                        if (! isExistIndex(list, vIndex2)) list.push(vIndex2);
                        normals.push( core.normal(i, point1) );
                        normals.push( core.normal(i, point2) );
                    }
                }
            }
            break;
        case POLY_MODE: // polygon
            var pCount = core.polygonCount();
            for (i = 0;i < pCount;i++) {
                if (core.polygonSelection(i)) {
                    var pSize = core.polygonSize(i);
                    var j;
                    for (j = 0;j < pSize;j++) {
                        var vIndex = core.vertexIndex(i,j);
                        if (! isExistIndex(list, vIndex)) list.push(vIndex);
                        normals.push( core.normal(i, j) );
                    }
                }
            }
            break;
    }
    
    var len = normals.length;
    if (len > 0) {
      normal = new Vec3D();
      for (i = 0;i < len;i++) {
        normal = normal.add( normals[i] );
      }
      
      normal = normal.multiply( 1 / len );
    }
    
    return list;
}

function isExistIndex(list, num) {
    var len = list.length;
    var i;
    for (i = 0;i < len;i++) {
        if (list[i] == num) return true;
    }
    return false;
}

function captureValue(tool) {
  var obj = tool.document().selectedObject();
  
  if (obj.type() == POLYGONOBJ) {
    var core = obj.core();
    var i;
    var vList = getVertexList(core, tool);
    var len = vList.length;
    if (len < 1) return;
    
    var center = core.vertex(vList[0]);
    for (i = 1;i < len;i++) {
      center = center.add( core.vertex(vList[i]) );
    }
    center = center.multiply( 1/ len );
    
    tool.setParameter("value x", center.x);
    tool.setParameter("value y", center.y);
    tool.setParameter("value z", center.z);
  }
}

function setValue(tool) {
  var obj = tool.document().selectedObject();
  var x = tool.getParameter("value x");
  var y = tool.getParameter("value y");
  var z = tool.getParameter("value z");
  
  var move = new Vec3D(x, y, z);
  
  if (tool.parameterWithName) obj.recordGeometryForUndo();
  
  if (obj.type() == POLYGONOBJ) {
    var core = obj.core();
    var i;
    var vList = getVertexList(core, tool);
    var len = vList.length;
    var center = core.vertex(vList[0]);
    for (i = 1;i < len;i++) {
      center = center.add( core.vertex(vList[i]) );
    }
    center = center.multiply( 1 / len );
    
    for (i = 0;i < len;i++) {
      var vec = core.vertex(vList[i]);
      var gap = vec.sub(center);
      
      vec = move.add( gap );
      
      core.setVertex(vList[i], vec);
    }
  }
  obj.update();
}

function Move(tool) {
    var doc = tool.document();
    var obj = doc.selectedObject();
    var coor = parseInt(tool.getParameter("coordinate system"));
    var matrix = obj.obj2WorldMatrix();
    var matrixBack = matrix.inverse();
    var xstep = (tool.getParameter("set x"))? tool.getParameter("step x"): 0;
    var ystep = (tool.getParameter("set y"))? tool.getParameter("step y"): 0;
    var zstep = (tool.getParameter("set z"))? tool.getParameter("step z"): 0;
    
    var move = new Vec3D(xstep, ystep, zstep);
    
    if (tool.parameterWithName) obj.recordGeometryForUndo();
    
    if (obj.type() == POLYGONOBJ) {
        var core = obj.core();
        var vCount = core.vertexCount();
        var i;
        var vList = getVertexList(core, tool);
        
        if (coor == 1) {
          var theta = Math.acos(normal.y)*180/Math.PI;
          var phi = Math.atan2(normal.x, normal.z)*180/Math.PI;
          
          var rotMat = new Mat4D( ROTATE_HPB, phi, theta, 0 );
          var rotMatBack = rotMat.inverse();
          if (vList.length > 0) {
            var centerVec = getCenterVecForSelector(coor, vList, obj, tool);
          } else {
            var centerVec = new Vec3D();
          }
        }
        //print(vList);
        var len = vList.length;
        for (i = 0;i < len;i++) {
            var vec = core.vertex(vList[i]);
            if (coor == 3) {
                vec = matrix.multiply(vec);
                vec = vec.add(move);
                vec = matrixBack.multiply(vec);
            } else if (coor == 1) {
                vec = vec.sub( centerVec );
                vec = rotMatBack.multiply( vec );
                vec = vec.add(move);
                vec = rotMat.multiply( vec );
                vec = vec.add( centerVec );
            } else {
                vec = vec.add(move);
            }
            core.setVertex(vList[i], vec);
            //print("Stepper:"+i);
        }
    }
    obj.update();
}

function Rotate(tool) {
    var doc = tool.document();
    var obj = doc.selectedObject();
    var coor = parseInt(tool.getParameter("coordinate system"));
    var matrix = obj.obj2WorldMatrix();
    var matrixBack = matrix.inverse();
    var xstep = (tool.getParameter("set x"))? tool.getParameter("step x"): 0;
    var ystep = (tool.getParameter("set y"))? tool.getParameter("step y"): 0;
    var zstep = (tool.getParameter("set z"))? tool.getParameter("step z"): 0;
    
    var rotate = new Mat4D(ROTATE,xstep,ystep,zstep);
    
    if (tool.parameterWithName) obj.recordGeometryForUndo();
    
    if (obj.type() == POLYGONOBJ) {
        var core = obj.core();
        var vCount = core.vertexCount();
        var i;
        var vList = getVertexList(core, tool);
        var len = vList.length;
        var centerVec = new Vec3D(0,0,0);
        if (len > 1) {
            centerVec = getCenterVecForSelector(coor, vList, obj, tool);
        }
        if (coor == 1) {
          var theta = Math.acos(normal.y)*180/Math.PI;
          var phi = Math.atan2(normal.x, normal.z)*180/Math.PI;
          
          var rotMat = new Mat4D( ROTATE_HPB, phi, theta, 0 );
          var rotMatBack = rotMat.inverse();
        }
        for (i = 0;i < len;i++) {
            var vec = core.vertex(vList[i]);
            if (coor == 3) {
                vec = matrix.multiply(vec);
                vec = rotate.multiply(vec);
                vec = matrixBack.multiply(vec);
            } else if (coor == 4) {
                vec = matrix.multiply(vec);
                vec = vec.sub(centerVec);
                vec = rotate.multiply(vec);
                vec = vec.add(centerVec);
                vec = matrixBack.multiply(vec);
            } else if (coor == 1) {
                vec = vec.sub(centerVec);
                vec = rotMatBack.multiply(vec);
                vec = rotate.multiply(vec);
                vec = rotMat.multiply(vec);
                vec = vec.add(centerVec);
            } else {
                vec = vec.sub(centerVec);
                vec = rotate.multiply(vec);
                vec = vec.add(centerVec);
            }
            core.setVertex(vList[i], vec);
        }
    }
    obj.update();
}

function Scale(tool) {
    var doc = tool.document();
    var obj = doc.selectedObject();
    var coor = parseInt(tool.getParameter("coordinate system"));
    var matrix = obj.obj2WorldMatrix();
    var matrixBack = matrix.inverse();
    var xstep = (tool.getParameter("set x"))? tool.getParameter("step x"): 1;
    var ystep = (tool.getParameter("set y"))? tool.getParameter("step y"): 1;
    var zstep = (tool.getParameter("set z"))? tool.getParameter("step z"): 1;
    
    var scale = new Mat4D(SCALE,xstep,ystep,zstep);
    
    if (tool.parameterWithName) obj.recordGeometryForUndo();
    
    if (obj.type() == POLYGONOBJ) {
        var core = obj.core();
        var vCount = core.vertexCount();
        var i;
        var vList = getVertexList(core, tool);
        var len = vList.length;
        var centerVec = new Vec3D(0,0,0);
        if (len > 1) {
            centerVec = getCenterVecForSelector(coor, vList, obj, tool);
        }
        if (coor == 1) {
          var theta = Math.acos(normal.y)*180/Math.PI;
          var phi = Math.atan2(normal.x, normal.z)*180/Math.PI;
          
          var rotMat = new Mat4D( ROTATE_HPB, phi, theta, 0 );
          var rotMatBack = rotMat.inverse();
        }
        for (i = 0;i < len;i++) {
            var vec = core.vertex(vList[i]);
            if (coor == 3) {
                vec = matrix.multiply(vec); // local -> global
                vec = scale.multiply(vec);
                vec = matrixBack.multiply(vec); // global -> local
            } else if (coor == 4) {
                vec = matrix.multiply(vec);
                vec = vec.sub(centerVec); // set center to 0
                vec = scale.multiply(vec);
                vec = vec.add(centerVec); // set back
                vec = matrixBack.multiply(vec);            
            } else if (coor == 1) {
                vec = vec.sub(centerVec);
                vec = rotMatBack.multiply( vec );
                vec = scale.multiply(vec);
                vec = rotMat.multiply( vec );
                vec = vec.add(centerVec);
            } else {
                vec = vec.sub(centerVec);
                vec = scale.multiply(vec);
                vec = vec.add(centerVec);
            }
            core.setVertex(vList[i], vec);
        }
    }
    obj.update();
}
// for Debugging.
function printVec3D(vec) {
    print('x:y:z-'+vec.x.toFixed(3)+':'+vec.y.toFixed(3)+':'+vec.z.toFixed(3));
}
function printMatrix(matrix) {
    print(matrix.m00+':'+matrix.m01+':'+matrix.m02+':'+matrix.m03);
    print(matrix.m10+':'+matrix.m11+':'+matrix.m12+':'+matrix.m13);
    print(matrix.m20+':'+matrix.m21+':'+matrix.m22+':'+matrix.m23);
    print(matrix.m30+':'+matrix.m31+':'+matrix.m32+':'+matrix.m33);    
}
