//
//  MarchingCube.m (dataset.m)
//
//  Marching Cubes
//
//  Created by Steve Challis on 25/01/2010.
//  Copyright 2010 All rights reserved.
//    from http://github.com/schallis/marchingcubes
//
//	Modified by Hiroto Tsubaki

#import "MarchingCube.h"

#import "mc_tables.h"

#include <stdlib.h>

@implementation MarchingCube

@synthesize isoValue, minValue, maxValue, triangles, num_triangles, num_allocated, num_elements, 
				tri_min_x, tri_min_y, tri_min_z, tri_max_x, tri_max_y, tri_max_z,
				min_x, min_y, min_z, max_x, max_y, max_z, gridSize;

-(void)clearData {
    free(triangles);
    [self setTriangles:NULL];
    [self setNum_triangles:0];
    [self setNum_allocated:0];
    [self setNum_elements:0];
    [self setNum_elements:0];
}

- ( id )initWithBytes:(char *)data gridSizeX:(int)sizex sizeY:(int)sizey sizeZ:(int)sizez blurIter:(int)blurIter useGCD: (bool)useGCD {
	
	self = [super init];
	
	if (self != nil) {
		dimensions[0] = sizex;
		dimensions[1] = sizey;
		dimensions[2] = sizez;
		
		_useGCD = useGCD;
		
		min_x = 0;
		min_y = 0;
		min_z = 0;
		
		int length = sizex * sizey * sizez;
		scalarData = (float *)malloc(sizeof( float )*length);
		
		if (blurIter < 1 || blurIter > 24) {
			blurIter = 1;
		}
		int blurDiv = blurIter*blurIter*blurIter;
		
		if (_useGCD) {
		 
			dispatch_queue_t queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0);
			dispatch_group_t group = dispatch_group_create();
			
			//NSLog(@"blurIter:%d, blurDiv:%d", blurIter, blurDiv);
			
			for (int i = 0;i < sizex;i++) {
				
				// dispatch 
				dispatch_group_async(group, queue, ^{
				
					for (int j = 0;j < sizey;j++) {
						for (int k =0;k < sizez;k++) {
							float density_d = 0.0f;
							int it, jt, kt;
							bool ic, jc, kc;
							
							for (int bi = 0;bi < blurIter;bi++) {
								for (int bj = 0;bj < blurIter;bj++) {
									for (int bk = 0;bk < blurIter;bk++) {
										it = (bi%2 != 0)? (int)ceil(bi/2)*-1 : (int)floor(bi/2);
										jt = (bj%2 != 0)? (int)ceil(bj/2)*-1 : (int)floor(bj/2);
										kt = (bk%2 != 0)? (int)ceil(bk/2)*-1 : (int)floor(bk/2);
										
										ic = (it < 0)? i > it *-1 : i < sizex - it ;
										jc = (jt < 0)? j > jt *-1 : j < sizey - jt ;
										kc = (kt < 0)? k > kt *-1 : k < sizez - kt ;
										
										if (ic && jc && kc) density_d += data[ (k+kt) + (j+jt)*sizez + (i+it)*sizez*sizey ];
									}
								}
							}
							scalarData[ k + j*sizez + i*sizez*sizey ] = density_d / blurDiv;
						}
					}
				});
				// dispatch
			}
			
			dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
			dispatch_release(group);
			
		} else {
			int it, jt, kt;
			bool ic, jc, kc;
			float density;
			 
			for (int i = 0;i < sizex;i++) {
				for (int j = 0;j < sizey;j++) {
					for (int k =0;k < sizez;k++) {
						density = 0.0f;
						
						for (int bi = 0;bi < blurIter;bi++) {
							for (int bj = 0;bj < blurIter;bj++) {
								for (int bk = 0;bk < blurIter;bk++) {
									it = (bi%2 != 0)? (int)ceil(bi/2)*-1 : (int)floor(bi/2);
									jt = (bj%2 != 0)? (int)ceil(bj/2)*-1 : (int)floor(bj/2);
									kt = (bk%2 != 0)? (int)ceil(bk/2)*-1 : (int)floor(bk/2);
									
									ic = (it < 0)? i > it *-1 : i < sizex - it ;
									jc = (jt < 0)? j > jt *-1 : j < sizey - jt ;
									kc = (kt < 0)? k > kt *-1 : k < sizez - kt ;
									
									if (ic && jc && kc) density += data[ (k+kt) + (j+jt)*sizez + (i+it)*sizez*sizey ];
								}
							}
						}
						scalarData[ k + j*sizez + i*sizez*sizey ] = density / blurDiv;
					}
				}
			}
		}
		
		[self setMaxValue:1.0f];
		[self setMinValue:0.0f];
	}
	return self;
}

- (float)isovalueFromUnitIsovalue:(float)value {
    // given a value between 0 and 1, return
    // corresponding isovalues for dataset
    return value * (maxValue - minValue) + minValue;
}

- (void)recalculateWithIsovalue:(float)value; {
	
    float normIso = [self isovalueFromUnitIsovalue:value];
	[self setIsoValue:normIso];
	
    int x = dimensions[0];
    int y = dimensions[1];
    int z = dimensions[2];
	
	int grid = (x > y)?
					(x > z)? x : z :
					(y > z)? y : z ;
	
	if (_useGCD) {
		
		dispatch_queue_t queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0);
		dispatch_group_t group = dispatch_group_create();
		
		dispatch_semaphore_t sema = dispatch_semaphore_create(1);
		
		// Loop over cells in dataset
		for (int i=0; i<x-1; i++) {
			
				for (int j=0; j<y-1; j++) {
					
					dispatch_group_async(group, queue, ^{
						
					for (int k=0; k<z-1; k++) {
						// Get surrounding vertices indexes
						int is[8] = {
							// Bits are in reverse order
							k + j*z + i*z*y,            // 000  0
							k + j*z + (i+1)*z*y,        // 001  1
							k + (j+1)*z + i*z*y,        // 010  2
							k + (j+1)*z + (i+1)*z*y,    // 011  3
							(k+1) + (j*z) + i*z*y,      // 100  4
							(k+1) + (j*z) + (i+1)*z*y,  // 101  5
							(k+1) + (j+1)*z + i*z*y,    // 110  6
							(k+1) + (j+1)*z + (i+1)*z*y // 111  7
						};
						float vs[8];
						for (int n=0; n<8; n++) {
							vs[n] = scalarData[is[n]];
						}
						// Figure out which ones are above the threshold
						// Label values 1 or 0 depending on whether they are above isovalue
						int b_vs[8];
						for (int n=0; n<8; n++) {
							b_vs[n] = vs[n] >= normIso;
						}
						// Get decimal value to lookup
						int lookup = b_vs[0] + (b_vs[1]*2) +
						(b_vs[2]*4) + (b_vs[3]*8) +
						(b_vs[4]*16) + (b_vs[5]*32) +
						(b_vs[6]*64) + (b_vs[7]*128);
						// Get triangles
						for (int t=0; t<triangleTable[lookup][0]; t++) {
							// Edges
							int edges[3] = {
								triangleTable[lookup][(t*3)+1],
								triangleTable[lookup][(t*3)+2],
								triangleTable[lookup][(t*3)+3]
							};
							// Get edge points
							int edgePoints[6];
							for (int pairs=0; pairs<3; pairs++) {
								for (int pair=0; pair<2; pair++) {
									edgePoints[(pairs*2)+pair] = edgeTable[edges[pairs]][pair];
								}
							};
							// Figure out triangle points
							
							
							Triangle triangle;
							for (int edge=0; edge<3; edge++) {
								// points at end of edges
								int point1 = edgePoints[2*edge];
								int point2 = edgePoints[(2*edge+1)];
								// Binary values, to find out which dimension to interpolate
								// Remember that the bits are reversed
								int p1[3] = {
									vertPos[edgePoints[2*edge]][2],
									vertPos[edgePoints[2*edge]][1],
									vertPos[edgePoints[2*edge]][0]};
								int p2[3] = {
									vertPos[edgePoints[(2*edge+1)]][2],
									vertPos[edgePoints[(2*edge+1)]][1],
									vertPos[edgePoints[(2*edge+1)]][0]};
								// coords of those points
								double c1 = scalarData[is[point1]];
								double c2 = scalarData[is[point2]];
								
								float interp = (isoValue-c1)/(c2-c1);
								double px = interp*abs(p1[0]-p2[0])+(p1[0]*p2[0]);
								double py = interp*abs(p1[1]-p2[1])+(p1[1]*p2[1]);
								double pz = interp*abs(p1[2]-p2[2])+(p1[2]*p2[2]);
								/*
								double point_pos[3] = {
									// Normalise to +-0.5
									((i+px)-((grid-1)/2.0))/(grid-1),
									((j+py)-((grid-1)/2.0))/(grid-1),
									((k+pz)-((grid-1)/2.0))/(grid-1)};
								*/
								double point_pos[3] = {
									min_x+(i+px)*gridSize,
									min_y+(j+py)*gridSize,
									min_z+(k+pz)*gridSize
								};
								triangle.points[edge][0] = point_pos[0];
								triangle.points[edge][1] = point_pos[1];
								triangle.points[edge][2] = point_pos[2];
								
								
								
								dispatch_semaphore_wait(sema, DISPATCH_TIME_FOREVER);
								// store min/max value
								if (i == 0 && k == 0 && j == 0 && t == 0 && edge == 0) {
									tri_min_x = point_pos[0];
									tri_min_y = point_pos[1];
									tri_min_z = point_pos[2];
									tri_max_x = point_pos[0];
									tri_max_y = point_pos[1];
									tri_max_z = point_pos[2];
								} else {
									if (point_pos[0] < tri_min_x) tri_min_x = point_pos[0];
									if (point_pos[1] < tri_min_y) tri_min_y = point_pos[1];
									if (point_pos[2] < tri_min_z) tri_min_z = point_pos[2];
									
									if (point_pos[0] > tri_max_x) tri_max_x = point_pos[0];
									if (point_pos[1] > tri_max_y) tri_max_y = point_pos[1];
									if (point_pos[2] > tri_max_z) tri_max_z = point_pos[2];
								}
								dispatch_semaphore_signal(sema);
								
							};
							dispatch_semaphore_wait(sema, DISPATCH_TIME_FOREVER);
							
							[self setNum_triangles:[self num_triangles]+1];
							[self addTriangle:triangle];
							
							dispatch_semaphore_signal(sema);
						}
					}
					});
				}
			
		}
		
		dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
		
		dispatch_release(group);
		
	} else {
		// Loop over cells in dataset
		for (int i=0; i<x-1; i++) {
			for (int j=0; j<y-1; j++) {
				for (int k=0; k<z-1; k++) {
					// Get surrounding vertices indexes
					int is[8] = {
						// Bits are in reverse order
						k + j*z + i*z*y,            // 000  0
						k + j*z + (i+1)*z*y,        // 001  1
						k + (j+1)*z + i*z*y,        // 010  2
						k + (j+1)*z + (i+1)*z*y,    // 011  3
						(k+1) + (j*z) + i*z*y,      // 100  4
						(k+1) + (j*z) + (i+1)*z*y,  // 101  5
						(k+1) + (j+1)*z + i*z*y,    // 110  6
						(k+1) + (j+1)*z + (i+1)*z*y // 111  7
					};
					float vs[8];
					for (int n=0; n<8; n++) {
						vs[n] = scalarData[is[n]];
					}
					// Figure out which ones are above the threshold
					// Label values 1 or 0 depending on whether they are above isovalue
					int b_vs[8];
					for (int n=0; n<8; n++) {
						b_vs[n] = vs[n] >= normIso;
					}
					// Get decimal value to lookup
					int lookup = b_vs[0] + (b_vs[1]*2) +
					(b_vs[2]*4) + (b_vs[3]*8) +
					(b_vs[4]*16) + (b_vs[5]*32) +
					(b_vs[6]*64) + (b_vs[7]*128);
					// Get triangles
					for (int t=0; t<triangleTable[lookup][0]; t++) {
						// Edges
						int edges[3] = {
							triangleTable[lookup][(t*3)+1],
							triangleTable[lookup][(t*3)+2],
							triangleTable[lookup][(t*3)+3]
						};
						// Get edge points
						int edgePoints[6];
						for (int pairs=0; pairs<3; pairs++) {
							for (int pair=0; pair<2; pair++) {
								edgePoints[(pairs*2)+pair] = edgeTable[edges[pairs]][pair];
							}
						};
						// Figure out triangle points
						Triangle triangle;
						for (int edge=0; edge<3; edge++) {
							// points at end of edges
							int point1 = edgePoints[2*edge];
							int point2 = edgePoints[(2*edge+1)];
							// Binary values, to find out which dimension to interpolate
							// Remember that the bits are reversed
							int p1[3] = {
								vertPos[edgePoints[2*edge]][2],
								vertPos[edgePoints[2*edge]][1],
								vertPos[edgePoints[2*edge]][0]};
							int p2[3] = {
								vertPos[edgePoints[(2*edge+1)]][2],
								vertPos[edgePoints[(2*edge+1)]][1],
								vertPos[edgePoints[(2*edge+1)]][0]};
							// coords of those points
							double c1 = scalarData[is[point1]];
							double c2 = scalarData[is[point2]];
							
							float interp = (isoValue-c1)/(c2-c1);
							double px = interp*abs(p1[0]-p2[0])+(p1[0]*p2[0]);
							double py = interp*abs(p1[1]-p2[1])+(p1[1]*p2[1]);
							double pz = interp*abs(p1[2]-p2[2])+(p1[2]*p2[2]);
							double point_pos[3] = {
								// Normalise to +-0.5
								((i+px)-((grid-1)/2.0))/(grid-1),
								((j+py)-((grid-1)/2.0))/(grid-1),
								((k+pz)-((grid-1)/2.0))/(grid-1)};
							triangle.points[edge][0] = point_pos[0];
							triangle.points[edge][1] = point_pos[1];
							triangle.points[edge][2] = point_pos[2];
							
							// store min/max value
							if (i == 0 && k == 0 && j == 0 && t == 0 && edge == 0) {
								tri_min_x = point_pos[0];
								tri_min_y = point_pos[1];
								tri_min_z = point_pos[2];
								tri_max_x = point_pos[0];
								tri_max_y = point_pos[1];
								tri_max_z = point_pos[2];
							} else {
								if (point_pos[0] < tri_min_x) tri_min_x = point_pos[0];
								if (point_pos[1] < tri_min_y) tri_min_y = point_pos[1];
								if (point_pos[2] < tri_min_z) tri_min_z = point_pos[2];
								
								if (point_pos[0] > tri_max_x) tri_max_x = point_pos[0];
								if (point_pos[1] > tri_max_y) tri_max_y = point_pos[1];
								if (point_pos[2] > tri_max_z) tri_max_z = point_pos[2];
							}
						};
						[self setNum_triangles:[self num_triangles]+1];
						[self addTriangle:triangle];
					}
				}
			}
		}
	}
}
    
// Dynamically resize array to accommodate new objects
-(int)addTriangle:(Triangle)item {

	if(num_elements == num_allocated) { // Are more refs required?
        
		if (num_allocated == 0)
			num_allocated = 10; // Start off with 10 refs
		else
			num_allocated *= 2; // Double the number of refs allocated
		
		// Make the reallocation transactional by using a temporary variable first
		void *_tmp = realloc([self triangles], (num_allocated * sizeof(Triangle)));
		
		// If the reallocation didn't go so well, inform the user and bail out
		if (!_tmp) { 
			fprintf(stderr, "ERROR: Couldn't realloc memory!\n");
			return(-1); 
		}
		[self setTriangles:(Triangle*)_tmp];	
	}
	
	triangles[num_elements] = item; 
	num_elements++;
	
	return num_elements;
}

-(void)dealloc {
    
	free(triangles);
	free(scalarData);
	
    [super dealloc];
}

@end
